<?php
/**
 * Plugin Name: Content Tracker Pro
 * Plugin URI: https://colorpack.co.th/free-extension/content-tracker.html
 * Description: ติดตามการสร้างเนื้อหา ดูว่าใครสร้าง Post/Page ไหนบ้าง พร้อมสถิติและ Export CSV
 * Version: 1.0.0
 * Author: Pisan Chueachatchai
 * Author URI: http://colorpack.co.th/
 * License: GPL v2 or later
 * Text Domain: content-tracker
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ContentTrackerPro {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_ct_get_content', array($this, 'ajax_get_content'));
        add_action('wp_ajax_ct_export_csv', array($this, 'ajax_export_csv'));
        add_action('wp_ajax_ct_get_stats', array($this, 'ajax_get_stats'));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Content Tracker',
            'Content Tracker',
            'manage_options',
            'content-tracker',
            array($this, 'render_admin_page'),
            'dashicons-chart-area',
            30
        );
    }
    
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_content-tracker') {
            return;
        }
        
        wp_enqueue_style(
            'content-tracker-style',
            plugin_dir_url(__FILE__) . 'assets/css/admin-style.css',
            array(),
            '1.0.0'
        );
        
        wp_enqueue_script(
            'content-tracker-script',
            plugin_dir_url(__FILE__) . 'assets/js/admin-script.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('content-tracker-script', 'contentTracker', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('content_tracker_nonce')
        ));
    }
    
    public function render_admin_page() {
        $authors = $this->get_all_authors();
        $post_types = get_post_types(array('public' => true), 'objects');
        ?>
        <div class="ct-wrap">
            <div class="ct-header">
                <div class="ct-header-content">
                    <h1>📊 Content Tracker Pro</h1>
                    <p class="ct-subtitle">ติดตามการสร้างเนื้อหาทั้งหมดในเว็บไซต์ของคุณ</p>
                </div>
            </div>
            
            <!-- Statistics Dashboard -->
            <div class="ct-stats-grid" id="ct-stats-grid">
                <div class="ct-stat-card ct-loading">
                    <div class="ct-stat-icon">📝</div>
                    <div class="ct-stat-number" id="stat-total-posts">-</div>
                    <div class="ct-stat-label">โพสต์ทั้งหมด</div>
                </div>
                <div class="ct-stat-card ct-loading">
                    <div class="ct-stat-icon">📄</div>
                    <div class="ct-stat-number" id="stat-total-pages">-</div>
                    <div class="ct-stat-label">เพจทั้งหมด</div>
                </div>
                <div class="ct-stat-card ct-loading">
                    <div class="ct-stat-icon">👥</div>
                    <div class="ct-stat-number" id="stat-total-authors">-</div>
                    <div class="ct-stat-label">ผู้เขียนทั้งหมด</div>
                </div>
                <div class="ct-stat-card ct-loading">
                    <div class="ct-stat-icon">📅</div>
                    <div class="ct-stat-number" id="stat-this-month">-</div>
                    <div class="ct-stat-label">เดือนนี้</div>
                </div>
            </div>
            
            <!-- Author Stats Chart -->
            <div class="ct-chart-section">
                <h2>📈 สถิติตามผู้เขียน</h2>
                <div class="ct-author-chart" id="ct-author-chart"></div>
            </div>
            
            <!-- Filters -->
            <div class="ct-filters">
                <div class="ct-filter-group">
                    <label for="ct-post-type">ประเภทเนื้อหา:</label>
                    <select id="ct-post-type">
                        <option value="all">ทั้งหมด</option>
                        <?php foreach ($post_types as $type): ?>
                            <option value="<?php echo esc_attr($type->name); ?>">
                                <?php echo esc_html($type->labels->singular_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="ct-filter-group">
                    <label for="ct-author">ผู้เขียน:</label>
                    <select id="ct-author">
                        <option value="all">ทั้งหมด</option>
                        <?php foreach ($authors as $author): ?>
                            <option value="<?php echo esc_attr($author->ID); ?>">
                                <?php echo esc_html($author->display_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="ct-filter-group">
                    <label for="ct-status">สถานะ:</label>
                    <select id="ct-status">
                        <option value="all">ทั้งหมด</option>
                        <option value="publish">เผยแพร่</option>
                        <option value="draft">ฉบับร่าง</option>
                        <option value="pending">รอตรวจสอบ</option>
                        <option value="private">ส่วนตัว</option>
                    </select>
                </div>
                
                <div class="ct-filter-group">
                    <label for="ct-date-from">ตั้งแต่:</label>
                    <input type="date" id="ct-date-from">
                </div>
                
                <div class="ct-filter-group">
                    <label for="ct-date-to">ถึง:</label>
                    <input type="date" id="ct-date-to">
                </div>
                
                <div class="ct-filter-group">
                    <label for="ct-search">ค้นหา:</label>
                    <input type="text" id="ct-search" placeholder="พิมพ์คำค้นหา...">
                </div>
                
                <div class="ct-filter-actions">
                    <button id="ct-filter-btn" class="ct-btn ct-btn-primary">
                        🔍 กรอง
                    </button>
                    <button id="ct-reset-btn" class="ct-btn ct-btn-secondary">
                        ↻ รีเซ็ต
                    </button>
                    <button id="ct-export-btn" class="ct-btn ct-btn-success">
                        📥 Export CSV
                    </button>
                </div>
            </div>
            
            <!-- Content Table -->
            <div class="ct-table-container">
                <div class="ct-table-header">
                    <h2>📋 รายการเนื้อหา</h2>
                    <div class="ct-pagination-info">
                        แสดง <span id="ct-showing">0</span> จาก <span id="ct-total">0</span> รายการ
                    </div>
                </div>
                
                <table class="ct-table" id="ct-content-table">
                    <thead>
                        <tr>
                            <th class="ct-th-id">#</th>
                            <th class="ct-th-title">หัวข้อ</th>
                            <th class="ct-th-type">ประเภท</th>
                            <th class="ct-th-author">ผู้เขียน</th>
                            <th class="ct-th-status">สถานะ</th>
                            <th class="ct-th-date">วันที่สร้าง</th>
                            <th class="ct-th-modified">แก้ไขล่าสุด</th>
                            <th class="ct-th-actions">จัดการ</th>
                        </tr>
                    </thead>
                    <tbody id="ct-table-body">
                        <tr class="ct-loading-row">
                            <td colspan="8">
                                <div class="ct-spinner"></div>
                                <span>กำลังโหลดข้อมูล...</span>
                            </td>
                        </tr>
                    </tbody>
                </table>
                
                <!-- Pagination -->
                <div class="ct-pagination" id="ct-pagination"></div>
            </div>
        </div>
        <?php
    }
    
    private function get_all_authors() {
        return get_users(array(
            'who' => 'authors',
            'has_published_posts' => true,
            'orderby' => 'display_name',
            'order' => 'ASC'
        ));
    }
    
    public function ajax_get_stats() {
        check_ajax_referer('content_tracker_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        global $wpdb;
        
        // Total posts
        $total_posts = wp_count_posts('post');
        $posts_count = $total_posts->publish + $total_posts->draft + $total_posts->pending + $total_posts->private;
        
        // Total pages
        $total_pages = wp_count_posts('page');
        $pages_count = $total_pages->publish + $total_pages->draft + $total_pages->pending + $total_pages->private;
        
        // Total authors
        $authors_count = count(get_users(array('who' => 'authors', 'has_published_posts' => true)));
        
        // This month
        $this_month = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_status IN ('publish', 'draft', 'pending', 'private')
            AND post_type IN ('post', 'page')
            AND MONTH(post_date) = %d 
            AND YEAR(post_date) = %d",
            date('n'),
            date('Y')
        ));
        
        // Author stats
        $author_stats = $wpdb->get_results(
            "SELECT u.display_name, COUNT(p.ID) as post_count 
            FROM {$wpdb->users} u 
            LEFT JOIN {$wpdb->posts} p ON u.ID = p.post_author 
            AND p.post_status IN ('publish', 'draft', 'pending', 'private')
            AND p.post_type IN ('post', 'page')
            GROUP BY u.ID 
            HAVING post_count > 0
            ORDER BY post_count DESC 
            LIMIT 10"
        );
        
        wp_send_json_success(array(
            'posts' => $posts_count,
            'pages' => $pages_count,
            'authors' => $authors_count,
            'this_month' => $this_month,
            'author_stats' => $author_stats
        ));
    }
    
    public function ajax_get_content() {
        check_ajax_referer('content_tracker_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $post_type = isset($_POST['post_type']) ? sanitize_text_field($_POST['post_type']) : 'all';
        $author = isset($_POST['author']) ? intval($_POST['author']) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'all';
        $date_from = isset($_POST['date_from']) ? sanitize_text_field($_POST['date_from']) : '';
        $date_to = isset($_POST['date_to']) ? sanitize_text_field($_POST['date_to']) : '';
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $per_page = 20;
        
        $args = array(
            'posts_per_page' => $per_page,
            'paged' => $page,
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        // Post type
        if ($post_type === 'all') {
            $args['post_type'] = get_post_types(array('public' => true), 'names');
        } else {
            $args['post_type'] = $post_type;
        }
        
        // Author
        if ($author > 0) {
            $args['author'] = $author;
        }
        
        // Status
        if ($status === 'all') {
            $args['post_status'] = array('publish', 'draft', 'pending', 'private');
        } else {
            $args['post_status'] = $status;
        }
        
        // Date range
        if (!empty($date_from) || !empty($date_to)) {
            $date_query = array();
            if (!empty($date_from)) {
                $date_query['after'] = $date_from;
            }
            if (!empty($date_to)) {
                $date_query['before'] = $date_to . ' 23:59:59';
            }
            $date_query['inclusive'] = true;
            $args['date_query'] = array($date_query);
        }
        
        // Search
        if (!empty($search)) {
            $args['s'] = $search;
        }
        
        $query = new WP_Query($args);
        $items = array();
        
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();
                $author_id = get_the_author_meta('ID');
                $post_type_obj = get_post_type_object(get_post_type());
                
                $items[] = array(
                    'id' => $post_id,
                    'title' => get_the_title(),
                    'type' => $post_type_obj ? $post_type_obj->labels->singular_name : get_post_type(),
                    'type_slug' => get_post_type(),
                    'author' => get_the_author(),
                    'author_id' => $author_id,
                    'author_avatar' => get_avatar_url($author_id, array('size' => 32)),
                    'status' => get_post_status(),
                    'status_label' => $this->get_status_label(get_post_status()),
                    'date' => get_the_date('Y-m-d H:i'),
                    'modified' => get_the_modified_date('Y-m-d H:i'),
                    'edit_url' => get_edit_post_link($post_id, 'raw'),
                    'view_url' => get_permalink($post_id)
                );
            }
        }
        
        wp_reset_postdata();
        
        wp_send_json_success(array(
            'items' => $items,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'current_page' => $page
        ));
    }
    
    private function get_status_label($status) {
        $labels = array(
            'publish' => 'เผยแพร่',
            'draft' => 'ฉบับร่าง',
            'pending' => 'รอตรวจสอบ',
            'private' => 'ส่วนตัว',
            'future' => 'กำหนดเวลา',
            'trash' => 'ถังขยะ'
        );
        return isset($labels[$status]) ? $labels[$status] : $status;
    }
    
    public function ajax_export_csv() {
        check_ajax_referer('content_tracker_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $post_type = isset($_POST['post_type']) ? sanitize_text_field($_POST['post_type']) : 'all';
        $author = isset($_POST['author']) ? intval($_POST['author']) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'all';
        $date_from = isset($_POST['date_from']) ? sanitize_text_field($_POST['date_from']) : '';
        $date_to = isset($_POST['date_to']) ? sanitize_text_field($_POST['date_to']) : '';
        $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
        
        $args = array(
            'posts_per_page' => -1,
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        // Post type
        if ($post_type === 'all') {
            $args['post_type'] = get_post_types(array('public' => true), 'names');
        } else {
            $args['post_type'] = $post_type;
        }
        
        // Author
        if ($author > 0) {
            $args['author'] = $author;
        }
        
        // Status
        if ($status === 'all') {
            $args['post_status'] = array('publish', 'draft', 'pending', 'private');
        } else {
            $args['post_status'] = $status;
        }
        
        // Date range
        if (!empty($date_from) || !empty($date_to)) {
            $date_query = array();
            if (!empty($date_from)) {
                $date_query['after'] = $date_from;
            }
            if (!empty($date_to)) {
                $date_query['before'] = $date_to . ' 23:59:59';
            }
            $date_query['inclusive'] = true;
            $args['date_query'] = array($date_query);
        }
        
        // Search
        if (!empty($search)) {
            $args['s'] = $search;
        }
        
        $query = new WP_Query($args);
        $csv_data = array();
        
        // Header
        $csv_data[] = array('ID', 'หัวข้อ', 'ประเภท', 'ผู้เขียน', 'สถานะ', 'วันที่สร้าง', 'แก้ไขล่าสุด', 'URL');
        
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $post_type_obj = get_post_type_object(get_post_type());
                
                $csv_data[] = array(
                    get_the_ID(),
                    get_the_title(),
                    $post_type_obj ? $post_type_obj->labels->singular_name : get_post_type(),
                    get_the_author(),
                    $this->get_status_label(get_post_status()),
                    get_the_date('Y-m-d H:i:s'),
                    get_the_modified_date('Y-m-d H:i:s'),
                    get_permalink()
                );
            }
        }
        
        wp_reset_postdata();
        
        // Generate CSV with proper UTF-8 encoding
        $output = fopen('php://temp', 'r+');
        
        // UTF-8 BOM for Excel to recognize Thai characters
        fwrite($output, "\xEF\xBB\xBF");
        
        foreach ($csv_data as $row) {
            fputcsv($output, $row);
        }
        
        rewind($output);
        $csv_content = stream_get_contents($output);
        fclose($output);
        
        // Send as direct download instead of base64
        wp_send_json_success(array(
            'csv' => $csv_content,
            'filename' => 'content-tracker-export-' . date('Y-m-d-His') . '.csv'
        ));
    }
}

// Initialize plugin
ContentTrackerPro::get_instance();
