/**
 * Content Tracker Pro - Admin JavaScript
 */

(function($) {
    'use strict';

    const CT = {
        currentPage: 1,
        totalPages: 1,
        isLoading: false,

        init: function() {
            this.loadStats();
            this.loadContent();
            this.bindEvents();
        },

        bindEvents: function() {
            $('#ct-filter-btn').on('click', () => {
                this.currentPage = 1;
                this.loadContent();
            });

            $('#ct-reset-btn').on('click', () => {
                this.resetFilters();
            });

            $('#ct-export-btn').on('click', () => {
                this.exportCSV();
            });

            // Enter key on search
            $('#ct-search').on('keypress', (e) => {
                if (e.which === 13) {
                    this.currentPage = 1;
                    this.loadContent();
                }
            });
        },

        loadStats: function() {
            $.ajax({
                url: contentTracker.ajaxurl,
                type: 'POST',
                data: {
                    action: 'ct_get_stats',
                    nonce: contentTracker.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.renderStats(response.data);
                    }
                },
                error: () => {
                    console.error('Failed to load stats');
                }
            });
        },

        renderStats: function(data) {
            $('#stat-total-posts').text(this.formatNumber(data.posts));
            $('#stat-total-pages').text(this.formatNumber(data.pages));
            $('#stat-total-authors').text(this.formatNumber(data.authors));
            $('#stat-this-month').text(this.formatNumber(data.this_month));

            $('.ct-stat-card').removeClass('ct-loading');

            // Render author chart
            this.renderAuthorChart(data.author_stats);
        },

        renderAuthorChart: function(authorStats) {
            if (!authorStats || authorStats.length === 0) {
                $('#ct-author-chart').html('<div class="ct-empty-state"><div class="ct-empty-state-icon">📊</div><div class="ct-empty-state-text">ไม่พบข้อมูลผู้เขียน</div></div>');
                return;
            }

            const maxCount = Math.max(...authorStats.map(a => parseInt(a.post_count)));
            let html = '';

            authorStats.forEach((author, index) => {
                const percentage = (parseInt(author.post_count) / maxCount) * 100;
                const colors = [
                    'linear-gradient(90deg, #6366f1 0%, #8b5cf6 100%)',
                    'linear-gradient(90deg, #3b82f6 0%, #06b6d4 100%)',
                    'linear-gradient(90deg, #22c55e 0%, #10b981 100%)',
                    'linear-gradient(90deg, #f59e0b 0%, #f97316 100%)',
                    'linear-gradient(90deg, #ef4444 0%, #f43f5e 100%)'
                ];

                html += `
                    <div class="ct-author-bar-container" style="animation-delay: ${index * 0.1}s">
                        <div class="ct-author-bar-header">
                            <span class="ct-author-bar-name">${this.escapeHtml(author.display_name)}</span>
                            <span class="ct-author-bar-count">${author.post_count} รายการ</span>
                        </div>
                        <div class="ct-author-bar-track">
                            <div class="ct-author-bar-fill" style="width: ${percentage}%; background: ${colors[index % colors.length]}"></div>
                        </div>
                    </div>
                `;
            });

            $('#ct-author-chart').html(html);
        },

        loadContent: function() {
            if (this.isLoading) return;
            this.isLoading = true;

            const filters = this.getFilters();
            filters.page = this.currentPage;

            $('#ct-table-body').html(`
                <tr class="ct-loading-row">
                    <td colspan="8">
                        <div class="ct-spinner"></div>
                        <span>กำลังโหลดข้อมูล...</span>
                    </td>
                </tr>
            `);

            $.ajax({
                url: contentTracker.ajaxurl,
                type: 'POST',
                data: {
                    action: 'ct_get_content',
                    nonce: contentTracker.nonce,
                    ...filters
                },
                success: (response) => {
                    if (response.success) {
                        this.renderContent(response.data);
                    } else {
                        this.showError('ไม่สามารถโหลดข้อมูลได้');
                    }
                    this.isLoading = false;
                },
                error: () => {
                    this.showError('เกิดข้อผิดพลาดในการเชื่อมต่อ');
                    this.isLoading = false;
                }
            });
        },

        getFilters: function() {
            return {
                post_type: $('#ct-post-type').val(),
                author: $('#ct-author').val(),
                status: $('#ct-status').val(),
                date_from: $('#ct-date-from').val(),
                date_to: $('#ct-date-to').val(),
                search: $('#ct-search').val()
            };
        },

        renderContent: function(data) {
            const { items, total, pages, current_page } = data;
            
            this.totalPages = pages;
            this.currentPage = current_page;

            $('#ct-showing').text(items.length);
            $('#ct-total').text(total);

            if (items.length === 0) {
                $('#ct-table-body').html(`
                    <tr>
                        <td colspan="8">
                            <div class="ct-empty-state">
                                <div class="ct-empty-state-icon">📭</div>
                                <div class="ct-empty-state-text">ไม่พบเนื้อหาตามเงื่อนไขที่เลือก</div>
                            </div>
                        </td>
                    </tr>
                `);
                $('#ct-pagination').html('');
                return;
            }

            let html = '';
            items.forEach((item, index) => {
                html += `
                    <tr style="animation-delay: ${index * 0.03}s">
                        <td><span class="ct-post-id">${item.id}</span></td>
                        <td><span class="ct-post-title">${this.escapeHtml(item.title || '(ไม่มีหัวข้อ)')}</span></td>
                        <td><span class="ct-post-type">${this.escapeHtml(item.type)}</span></td>
                        <td>
                            <div class="ct-author-cell">
                                <img src="${item.author_avatar}" alt="${this.escapeHtml(item.author)}" class="ct-author-avatar">
                                <span class="ct-author-name">${this.escapeHtml(item.author)}</span>
                            </div>
                        </td>
                        <td><span class="ct-status-badge ct-status-${item.status}">${item.status_label}</span></td>
                        <td><span class="ct-date-cell">${item.date}</span></td>
                        <td><span class="ct-date-cell">${item.modified}</span></td>
                        <td>
                            <div class="ct-actions">
                                <a href="${item.edit_url}" class="ct-action-btn ct-action-edit" target="_blank">✏️ แก้ไข</a>
                                <a href="${item.view_url}" class="ct-action-btn ct-action-view" target="_blank">👁️ ดู</a>
                            </div>
                        </td>
                    </tr>
                `;
            });

            $('#ct-table-body').html(html);
            this.renderPagination();
        },

        renderPagination: function() {
            if (this.totalPages <= 1) {
                $('#ct-pagination').html('');
                return;
            }

            let html = '';
            
            // Previous button
            html += `<button class="ct-page-btn" data-page="${this.currentPage - 1}" ${this.currentPage === 1 ? 'disabled' : ''}>← ก่อนหน้า</button>`;

            // Page numbers
            const startPage = Math.max(1, this.currentPage - 2);
            const endPage = Math.min(this.totalPages, this.currentPage + 2);

            if (startPage > 1) {
                html += `<button class="ct-page-btn" data-page="1">1</button>`;
                if (startPage > 2) {
                    html += `<span style="padding: 8px;">...</span>`;
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                html += `<button class="ct-page-btn ${i === this.currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
            }

            if (endPage < this.totalPages) {
                if (endPage < this.totalPages - 1) {
                    html += `<span style="padding: 8px;">...</span>`;
                }
                html += `<button class="ct-page-btn" data-page="${this.totalPages}">${this.totalPages}</button>`;
            }

            // Next button
            html += `<button class="ct-page-btn" data-page="${this.currentPage + 1}" ${this.currentPage === this.totalPages ? 'disabled' : ''}>ถัดไป →</button>`;

            $('#ct-pagination').html(html);

            // Bind pagination events
            $('.ct-page-btn').on('click', (e) => {
                const page = parseInt($(e.target).data('page'));
                if (page && page !== this.currentPage && page >= 1 && page <= this.totalPages) {
                    this.currentPage = page;
                    this.loadContent();
                    // Scroll to table
                    $('html, body').animate({
                        scrollTop: $('.ct-table-container').offset().top - 50
                    }, 300);
                }
            });
        },

        resetFilters: function() {
            $('#ct-post-type').val('all');
            $('#ct-author').val('all');
            $('#ct-status').val('all');
            $('#ct-date-from').val('');
            $('#ct-date-to').val('');
            $('#ct-search').val('');
            this.currentPage = 1;
            this.loadContent();
        },

        exportCSV: function() {
            const filters = this.getFilters();
            const $btn = $('#ct-export-btn');
            const originalText = $btn.html();

            $btn.html('⏳ กำลัง Export...').prop('disabled', true);

            $.ajax({
                url: contentTracker.ajaxurl,
                type: 'POST',
                data: {
                    action: 'ct_export_csv',
                    nonce: contentTracker.nonce,
                    ...filters
                },
                success: (response) => {
                    if (response.success) {
                        // Create blob with UTF-8 encoding for Thai support
                        const BOM = '\uFEFF';
                        const csvContent = response.data.csv;
                        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
                        const link = document.createElement('a');
                        const url = URL.createObjectURL(blob);
                        
                        link.setAttribute('href', url);
                        link.setAttribute('download', response.data.filename);
                        link.style.visibility = 'hidden';
                        document.body.appendChild(link);
                        link.click();
                        document.body.removeChild(link);
                        
                        // Clean up
                        URL.revokeObjectURL(url);

                        this.showSuccess('Export สำเร็จ!');
                    } else {
                        this.showError('ไม่สามารถ Export ได้');
                    }
                    $btn.html(originalText).prop('disabled', false);
                },
                error: () => {
                    this.showError('เกิดข้อผิดพลาดในการ Export');
                    $btn.html(originalText).prop('disabled', false);
                }
            });
        },

        showError: function(message) {
            this.showNotification(message, 'error');
        },

        showSuccess: function(message) {
            this.showNotification(message, 'success');
        },

        showNotification: function(message, type) {
            const bgColor = type === 'success' ? '#22c55e' : '#ef4444';
            const $notification = $(`
                <div style="
                    position: fixed;
                    top: 50px;
                    right: 20px;
                    background: ${bgColor};
                    color: white;
                    padding: 16px 24px;
                    border-radius: 8px;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                    z-index: 9999;
                    font-weight: 500;
                    animation: fadeIn 0.3s ease;
                ">
                    ${type === 'success' ? '✅' : '❌'} ${message}
                </div>
            `);

            $('body').append($notification);

            setTimeout(() => {
                $notification.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 3000);
        },

        formatNumber: function(num) {
            return new Intl.NumberFormat('th-TH').format(num || 0);
        },

        escapeHtml: function(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        CT.init();
    });

})(jQuery);
